/////////////////////////////////////////////////////////////////////////////////
// License: Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
/////////////////////////////////////////////////////////////////////////////////
//
// This shader is based on e#21916.3 from GlslSandbox.com,  the original author
// is not known.  It  has been rewritten by TheEmu for use with VGHD and at the
// same time both slightly tidied up and made more easily configurable.
//
/////////////////////////////////////////////////////////////////////////////////

// Shader inputs.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

uniform int depth;          // The depth into the scene

/////////////////////////////////////////////////////////////////////////////////

// The number of swarm size and its position are controlled by  the  following
// set of parameters. Normaly these will all be constants,  but  they may also
// be expressions involving shader variables visible where the parameters  are
// used and hence may depend on time or position.

#define SWARM_SIZE 20    // Number of elements in the swarm.

#define SWARM_CENTER 0.0 // Horizontal position of the swarm's center.
#define SWARM_RADIUS 0.5 // Width of the swarm, 1.0 = half screen width.

#define SWARM_BASE 0.1   // Base level of the swarm, 0 = bottom of screen.
#define SWARM_TOP  0.9   // Top level of the swarm,  1 = top of screen.

// The colour of the swarm elements and the way in which their glow fades with
// with distance from their individual centers are controled by the  following
// set of parameters.  The arguments i and r are the element index, which runs
// from 0 to SWARM_SIZE-1,  and  the position of the current pixel relative to
// the center of the element.

// In this particular example the swarm elements are heart shaped.

const float Pi = 3.1415926535897932384626433832795;

const float K0 = 3.0;
const float K1 = 4.4/Pi;
const float K2 = 2.0/(Pi*Pi);
const float K3 = 1.2*Pi;

float heart(vec2 r)
 { float h = abs(atan(r.x,r.y));
   return ( h / (K3-h) ) * ( h * ( h*K2 - K1 ) + K0 );
 }

#define SWARM_COLOUR(i,r)  vec3(0.0008,0.0,0.0)
#define SWARM_ELEMENT(i,r) SWARM_COLOUR(i,r) * heart(r) / pow(length(r),2.0)

// The way in which the swarm circulates are controlled by the  following  set
// of parameters.  See the main body of the shader for their meanings.  It  is
// possible to specify a different set of these parameters for each element of
// the swarm, i identifies the element and runs from 0 to SWARM_SIZE-1.

#define ALPHA(i) 1.00 // Time scale factor for the angle alpha.
#define BETA(i)  0.01 // Time scale factor for the angle beta.
#define GAMMA(i) 90.0 // Scale factor for gamma.
#define THETA(i) 0.00 // Initial value for the theta angle.

// Normally this shader is used twice, once to generate that part of the swarm
// that is in behind the VGHD clip and a second time to generate the part that
// is in front of it. The simplest way to select which elements to generate is
// to make he decision based on their theta angle which this is positive for a
// half of the orbit and negative for the other half.

#define GENERATE_ELEMENT(i,theta) (depth==1) ^^ (cos(theta) < 0.0)

// The following parameters modify the x and y coordinates. There is no reason
// to define them as non zero constants as SWARM_CENTER and SWARM_BASE already
// cover that use.  However they may be defined as expressions involving time,
// coordinates or internal variables of the shader such as theta to cause  the
// swarm elements to bob up and down or side to side as they orbit.

// This particular pair of DELTA parameters cause the swarm elements to bob up
// and down sinusoidaly with those in the middle bobbing most strongly and for
// the orbits to be wide in the middle and smaller at the top and bottom.

#define DELTA_X pow(abs(y-0.5)*2.0-0.1,4.0)*sin(theta)
#define DELTA_Y sin(theta*(0.503-abs(y-0.5))*11.0)/10.0

/////////////////////////////////////////////////////////////////////////////////
//
// This shader generates a swarm of glowing spheres which may be considered as
// rotating around around a cylinder with the center of each at a fixed height
// above the cylinder's base and at an angle theta around its circumference.
//
// Theta is given by
//
//    theta = THETA + ALPHA*t + sin(BETA*t)*GAMMA
//
// where t is the time and the other parameters are, nominaly,  constants.  As
// a result the whole swarm circles the cylinder at a speed  given  by  ALPHA,
// but the relative phases of each element varies depending on the  values  of
// BETA and GAMMA.  The result is that at times the swarm elements appear as a
// single starnd of lights,  at others as two or more strands and at yet other
// times there are no discernable strands.
//
// The shader is controlled by a number of parameters,  including ALPHA,  BETA
// and GAMMA, which are described above.  Although nominaly constant the value
// of any parameter can be defined as a function of time or position  allowing
// a great deal of flexibility. For example the spheres may be made to pulsate
// or their colours may be made to vary as time progresses.
//
/////////////////////////////////////////////////////////////////////////////////

void main( void )
 {

   // Get the extent of the swarm in the X and Y directions.

   vec2 swarm_scale = vec2 ( SWARM_RADIUS, SWARM_TOP - SWARM_BASE );

   // Get coordinates of the current pixel such that X increases from  X  from
   // left to right from -1.0 at the left edge to +1.0 at the right  edge  and
   // Y increases from 0.0 at the bottom to 1.0 at the top of the screen.

   vec2 pixel = gl_FragCoord.xy/u_WindowSize.xy;
   pixel.x = ( pixel.x - 0.5 ) * 2.0;

   // Adjust the X coordinate to be centered with the swarm and go  from  -1.0
   // to +1.0 at the edges of the swarm. Working in this coordinate system  is
   // slightly simpler than working in the screen coordinates themselves.  The
   // Y coordinate is similarly adjusted so that 0.0 corresponds to the bottom
   // of the swarm and 1.0 to its top.

   pixel.x = pixel.x / swarm_scale.x + SWARM_CENTER;
   pixel.y = ( pixel.y - SWARM_BASE ) / swarm_scale.y;

   // Initialise the colour for this pixel.

   vec3 colour = vec3(0.0);

   // Process each swarm element in turn.

   for ( int i = 0; i < SWARM_SIZE; i++ )
    {
      // Calcualate the alpha, beta and gamma angles.

      float alpha = ALPHA(i) * u_Elapsed;
      float beta  = BETA(i)  * u_Elapsed;
      float gamma = GAMMA(i) * sin(beta);

      // Get swarm element i's y coordinate and theta angle.

      float y = float(i) / float(SWARM_SIZE-1);
      float theta = THETA(i) + alpha + gamma*y;

      // If this swarm element is to be generated then do so.

      if ( GENERATE_ELEMENT(i,theta) )
       {
         // Get the distance between the element and the  pixel.
         // Multiplying by swarm_scale compensates for the swarm
         // and the screen coordinate systems  having  different
         // scales for each of their axes. If it is not done the
         // swarm elements become elongated along X or Y.

         vec2 r = pixel - vec2 ( sin(theta), y );
         r = r + vec2 ( DELTA_X, DELTA_Y );
         r = r * swarm_scale;

         // Add the contribution this element makes to the colour
         // of the current pixel.  The  effect is a glow centered
         // on the element with its shape and the  way  it  glows
         // controlled by SWARM_ELEMENT.

         colour += SWARM_ELEMENT(i,r);

       }

    }

   // Output the colour for the pixel.  By  using the "length" of
   // the colour as the value for the alpha component pixels  are
   // increasingly transparent the further away they are from any
   // swarm element - allowing any background to show through.

   gl_FragColor = vec4 ( colour, length(colour) );

 }